/**
 * Room Booking JavaScript
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        
        // Initialize date pickers
        if (typeof roomBookingData !== 'undefined') {
            initializeDatePickers();
            initializePriceCalculation();
            initializeBookingForm();
        }

        // Initialize Swiper for room gallery
        if (typeof Swiper !== 'undefined') {
            initializeRoomGallery();
        }
    });

    /**
     * Initialize Date Pickers
     */
    function initializeDatePickers() {
        var bookedDates = roomBookingData.bookedDates || [];
        var minStay = roomBookingData.minStay || 1;
        var maxStay = roomBookingData.maxStay || 0;
        
        // Convert booked dates to Flatpickr format
        var disabledDates = bookedDates.map(function(date) {
            return date;
        });

        var checkInPicker = flatpickr('#check_in_date', {
            minDate: 'today',
            dateFormat: 'Y-m-d',
            disable: disabledDates,
            altInput: false,
            onChange: function(selectedDates, dateStr, instance) {
                if (selectedDates.length > 0) {
                    var minDate = new Date(selectedDates[0]);
                    minDate.setDate(minDate.getDate() + minStay);
                    
                    if (checkOutPicker) {
                        checkOutPicker.set('minDate', minDate);
                        if (!checkOutPicker.input.value) {
                            checkOutPicker.setDate(minDate, false);
                        }
                    }
                    
                    calculatePrice();
                }
            }
        });

        var checkOutMinDate = new Date();
        checkOutMinDate.setDate(checkOutMinDate.getDate() + minStay);

        var checkOutPicker = flatpickr('#check_out_date', {
            minDate: checkOutMinDate,
            dateFormat: 'Y-m-d',
            disable: disabledDates,
            altInput: false,
            onChange: function() {
                calculatePrice();
            }
        });

        // Store pickers globally
        window.checkInPicker = checkInPicker;
        window.checkOutPicker = checkOutPicker;
    }

    /**
     * Initialize Price Calculation
     */
    function initializePriceCalculation() {
        $('#check_in_date, #check_out_date, #adults, #children').on('change', function() {
            calculatePrice();
        });

        $('.extra-service-checkbox').on('change', function() {
            calculatePrice();
        });

        $('input[name="payment_type"]').on('change', function() {
            calculatePrice();
        });
    }

    /**
     * Calculate Booking Price
     */
    function calculatePrice() {
        var checkIn = $('#check_in_date').val();
        var checkOut = $('#check_out_date').val();
        var adults = parseInt($('#adults').val()) || 0;
        var children = parseInt($('#children').val()) || 0;
        var roomId = roomBookingData.roomId;
        var extraServices = [];

        $('.extra-service-checkbox:checked').each(function() {
            extraServices.push($(this).val());
        });

        if (!checkIn || !checkOut) {
            return;
        }

        // Validate dates
        var checkInDate = new Date(checkIn);
        var checkOutDate = new Date(checkOut);
        
        if (checkOutDate <= checkInDate) {
            showMessage('Check-out date must be after check-in date.', 'error');
            return;
        }

        // Check minimum stay
        var nights = Math.ceil((checkOutDate - checkInDate) / (1000 * 60 * 60 * 24));
        if (nights < roomBookingData.minStay) {
            showMessage('Minimum stay is ' + roomBookingData.minStay + ' nights.', 'error');
            return;
        }

        // AJAX call to calculate price
        $.ajax({
            url: roomBooking.ajaxurl,
            type: 'POST',
            data: {
                action: 'calculate_booking_price',
                nonce: roomBooking.nonce,
                room_id: roomId,
                check_in: checkIn,
                check_out: checkOut,
                adults: adults,
                children: children,
                extra_services: extraServices
            },
            success: function(response) {
                if (response.success) {
                    updatePriceDisplay(response.data);
                } else {
                    showMessage(response.data.message || 'Error calculating price.', 'error');
                }
            },
            error: function() {
                showMessage('Error calculating price. Please try again.', 'error');
            }
        });
    }

    /**
     * Update Price Display
     */
    function updatePriceDisplay(data) {
        $('#nights-count').text(data.nights);
        $('#base-price').text(formatPrice(data.base_price));
        $('#total-price').text(data.formatted_total);

        // Extra services
        if (data.extra_services_price > 0) {
            $('#extra-services-price').text(formatPrice(data.extra_services_price));
            $('#extra-services-row').show();
        } else {
            $('#extra-services-row').hide();
        }

        // Payment options
        var paymentType = $('input[name="payment_type"]:checked').val();
        if (paymentType === 'deposit') {
            $('#deposit-amount').text(data.formatted_deposit);
            $('#remaining-balance').text(data.formatted_remaining);
            $('#deposit-info').show();
        } else {
            $('#deposit-info').hide();
        }
    }

    /**
     * Format Price
     */
    function formatPrice(price) {
        var symbol = roomBookingData.currencySymbol || '$';
        var position = roomBookingData.currencyPosition || 'left';
        
        if (position === 'left') {
            return symbol + parseFloat(price).toFixed(2);
        } else {
            return parseFloat(price).toFixed(2) + symbol;
        }
    }

    /**
     * Initialize Booking Form
     */
    function initializeBookingForm() {
        $('#room-booking-form').on('submit', function(e) {
            e.preventDefault();

            var checkIn = $('#check_in_date').val();
            var checkOut = $('#check_out_date').val();
            var adults = parseInt($('#adults').val()) || 0;
            var children = parseInt($('#children').val()) || 0;

            if (!checkIn || !checkOut) {
                showMessage('Please select check-in and check-out dates.', 'error');
                return false;
            }

            // Validate dates
            var checkInDate = new Date(checkIn);
            var checkOutDate = new Date(checkOut);
            
            if (checkOutDate <= checkInDate) {
                showMessage('Check-out date must be after check-in date.', 'error');
                return false;
            }

            // Check minimum stay
            var nights = Math.ceil((checkOutDate - checkInDate) / (1000 * 60 * 60 * 24));
            var minStay = roomBookingData.minStay || 1;
            if (nights < minStay) {
                showMessage('Minimum stay is ' + minStay + ' nights.', 'error');
                return false;
            }

            // Validate guest count
            var maxGuests = roomBookingData.maxGuests || 10;
            var totalGuests = adults + children;
            if (totalGuests > maxGuests) {
                showMessage('Maximum ' + maxGuests + ' guests allowed.', 'error');
                return false;
            }

            if (totalGuests < 1) {
                showMessage('Please select at least 1 guest.', 'error');
                return false;
            }

            // Add to cart - use standard form submission
            // The form will submit normally and WooCommerce will handle it
            // We just need to ensure validation passes
            var form = this;
            
            // If WooCommerce AJAX is available, use it
            if (typeof wc_add_to_cart_params !== 'undefined') {
                var formData = $(this).serialize();
                
                $.ajax({
                    url: wc_add_to_cart_params.wc_ajax_url.toString().replace('%%endpoint%%', 'add_to_cart'),
                    type: 'POST',
                    data: formData,
                    success: function(response) {
                        if (response.error && response.product_url) {
                            window.location = response.product_url;
                            return;
                        }
                        // Redirect to cart
                        if (typeof wc_add_to_cart_params !== 'undefined' && wc_add_to_cart_params.cart_url) {
                            window.location.href = wc_add_to_cart_params.cart_url;
                        } else {
                            window.location.href = '/cart/';
                        }
                    },
                    error: function() {
                        // Fallback to standard form submission
                        form.submit();
                    }
                });
            } else {
                // Fallback to standard form submission
                form.submit();
            }
        });

        // Inquiry button
        $('#inquiry-btn').on('click', function() {
            $('#inquiry-modal').modal('show');
        });

        // Inquiry form
        $('#inquiry-form').on('submit', function(e) {
            e.preventDefault();
            
            var formData = $(this).serialize();
            formData += '&action=send_room_inquiry';

            $.ajax({
                url: roomBooking.ajaxurl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        showMessage('Your inquiry has been sent successfully!', 'success');
                        $('#inquiry-form')[0].reset();
                        $('#inquiry-modal').modal('hide');
                    } else {
                        showMessage(response.data.message || 'Error sending inquiry.', 'error');
                    }
                },
                error: function() {
                    showMessage('Error sending inquiry. Please try again.', 'error');
                }
            });
        });
    }

    /**
     * Initialize Room Gallery
     */
    function initializeRoomGallery() {
        var mainSlider = new Swiper('.room-main-slider', {
            spaceBetween: 10,
            navigation: {
                nextEl: '.swiper-button-next',
                prevEl: '.swiper-button-prev',
            },
            pagination: {
                el: '.swiper-pagination',
                clickable: true,
            },
            thumbs: {
                swiper: {
                    el: '.room-thumbs-slider',
                    slidesPerView: 4,
                    spaceBetween: 10,
                }
            }
        });
    }

    /**
     * Show Message
     */
    function showMessage(message, type) {
        var messageClass = type === 'error' ? 'alert-danger' : 'alert-success';
        var html = '<div class="alert ' + messageClass + ' alert-dismissible fade show" role="alert">' +
                   message +
                   '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>' +
                   '</div>';
        
        $('#booking-messages').html(html);
        
        setTimeout(function() {
            $('.alert').fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }

})(jQuery);
